
import org.apache.commons.codec.binary.Base64;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.*;

import org.json.JSONArray;
import org.json.JSONObject;

public class Restapi {
    private static final int CHUNK_SIZE = 1024 * 1024;
    private String _username;
    private String _password;
    private String _url;
    private String _method;
    private Date _date;
    private String _contentMd5;
    private Boolean _async = false;
    private int _checkAction = 0;
    Map<String, String> headers;

    private final String METHOD_HEAD = "HEAD";
    private final String METHOD_GET = "GET";
    private final String METHOD_PUT = "PUT";
    private final String METHOD_DELETE = "DELETE";
    private final String METHOD_PATCH = "PATCH";
    private final String METHOD_POST = "POST";

    public Restapi(String username, String password) {
        _username = username;
        _password = password;
        allowMethods(METHOD_PATCH);//设置HttpURLConnection支持PATCH请求
    }

    /**
     * 设置认证方式
     * @param checkAction   0是基础认证 1是签名认证
     * @return
     */
    public Restapi setCheckAction(int checkAction) {
        _checkAction = checkAction;
        return this;
    }

    /**
     * 设置请求url
     * @param url
     * @return
     */
    public Restapi setUrl(String url) {
        _url = url;
        return this;
    }

    /**
     * 设置请求方法
     * @param method
     * @return
     */
    public Restapi setMethod(String method) {
        _method = method;
        return this;
    }

    /**
     * 设置Date头
     * @param date
     * @return
     */
    public Restapi setDate(Date date) {
        _date = date;
        return this;
    }

    /**
     * 设置md5
     * @param md5
     * @return
     */
    public Restapi setContentMd5(String md5) {
        _contentMd5 = md5;
        return this;
    }

    /**
     * 设置是否异步
     * @param async
     * @return
     */
    public Restapi setAsync(Boolean async) {
        _async = async;
        return this;
    }

    /**
     * 获取认证鉴权
     * @return
     * @throws IOException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public String getAuthorization() throws IOException, NoSuchAlgorithmException, InvalidKeyException {
        if (_checkAction == 0) {
            return "Basic " + Base64.encodeBase64String((_username + ":" + _password).getBytes());
        }

        URL url = new URL(_url);
        String uri = url.getPath();
        String query = url.getQuery();
        if (query != null) {
            uri += "?" + query;
        }

        String date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(_date);
        String data = _method + "&" + uri + "&" + date;
        if (_contentMd5 != null) {
            data += "&" + _contentMd5;
        }
        String password = Base64.encodeBase64String(_password.getBytes());
        Mac sha1_HMAC = Mac.getInstance("HmacSHA1");
        SecretKeySpec secretKeySpec = new SecretKeySpec(password.getBytes(), "HmacSHA1");
        sha1_HMAC.init(secretKeySpec);
        String hash = Base64.encodeBase64String(sha1_HMAC.doFinal(data.getBytes()));
        return " WESTYUN " + _username + ":" + hash;
    }

    /**
     * 获取文件（目录）列表
     * @return
     * @throws Exception
     */
    public String getList(){
        return getList(0, 100, "desc");
    }

    /**
     * 获取文件（目录）列表
     * @param iter   分页开始位置
     * @param limit  获取的文件数量，默认 100，最大 10000
     * @param order  asc 或 desc，按文件名升序或降序排列。默认 asc
     * @return
     */
    public String getList(int iter, int limit, String order) {
        Map<String, String> params = new HashMap<>();
        if (iter > 0) {
            params.put("x-list-iter", iter + "");
        }
        if (limit < 100) {
            limit = 100;
        }
        if (limit > 10000) {
            limit = 10000;
        }
        params.put("x-list-limit", limit + "");
        params.put("x-list-order", order);
        return fssCurl(params);
    }

    /**
     * 获取文件内容
     * @return
     * @throws Exception
     */
    public String getContent(){
        return getContent(0, 0, "UTF-8");
    }

    /**
     * 获取文件内容
     * @param charset    输出字符集
     * @return
     * @throws Exception
     */
    public String getContent(String charset){
        return getContent(0, 0, charset);
    }

    /**
     * 获取文件内容
     * @param start     开始位置，为0表示从文件头开始
     * @param end       结束位置，为0 表示获取到文件尾部
     * @param charset   输出字符集
     * @return
     */
    public String getContent(int start, int end, String charset) {
        Map<String, String> params = new HashMap<>();
        if (end == 0) {
            params.put("Range", "byte=" + start + "-");
        } else {
            params.put("Range", "byte=" + start + "-" + end);
        }
        return fssCurl(params, charset);
    }

    /**
     * 获取文件信息
     * @return
     */
    public String getFileInfo() {
        Map<String, String> params = new HashMap<>();
        return fssCurl(params);
    }

    /**
     * 复制或移动文件
     * @param source   源文件在fss中路径
     * @param type      0复制 1 移动
     * @return
     */
    public String copyOrMoveFile(String source, int type) {
        return copyOrMoveFile(source, type, null, null);
    }

    /**
     * 复制或移动文件
     * @param source     源文件在fss中路径
     * @param type       0复制 1 移动
     * @param directive  对x-west-metadata-x操作的方式
     * @param meta       文件元数据
     * @return
     */
    public String copyOrMoveFile(String source, int type, String directive, Map<String, String> meta) {
        Map<String, String> params = new HashMap<>();
        if (type == 0) {
            params.put("x-west-copy-source", source);
        } else {
            params.put("x-west-move-source", source);
        }

        if (!empty(directive)) {
            params.put("x-west-metadata-directive", directive);
        }

        if (meta != null) {
            for (Map.Entry<String, String> entry : meta.entrySet()) {
                params.put("x-west-meta-" + entry.getKey(), entry.getValue());
            }
        }

        return fssCurl(params);
    }

    /**
     * 创建目录
     * @return
     */
    public String createFolder() {
        Map<String, String> params = new HashMap<>();
        params.put("folder", "true");
        return fssCurl(params);
    }

    /**
     * 删除目录或文件
     * @return
     */
    public String deleteFolderOrFlie() {
        Map<String, String> params = new HashMap<>();
        if (_async) {
            params.put("x-west-async", "true");
        }

        return fssCurl(params);
    }

    /**
     * 设置文件元数据
     * @param meta  元数据
     * @return
     */
    public String setMetadata(Map<String, String> meta) {
        Map<String, String> params = new HashMap<>();
        for (Map.Entry<String, String> entry : meta.entrySet()) {
            params.put("x-west-meta-" + entry.getKey(), entry.getValue());
        }
        return  fssCurl(params);
    }

    /**
     * 小文件上传
     * @param path       文件本地路径
     * @return
     * @throws Exception
     */
    public String uploadFile(String path) throws IOException,NoSuchAlgorithmException,InvalidKeyException ,FssException {
        return uploadFile(path, null, null, 0, null, null);
    }

    /**
     * 小文件上传
     * @param path         文件本地路径
     * @param contentType  文件mime类型
     * @param secret       文件访问密码
     * @param ttl          文件元信息, 指定文件的生存时间，单位天，最大支持180天，不设置为长期保留
     * @param thumb        图片预处理参数
     * @param meta         文件元信息
     * @return
     * @throws Exception
     */
    public String uploadFile(String path, String contentType, String secret, int ttl, String thumb, Map<String, String> meta)
        throws IOException,NoSuchAlgorithmException,InvalidKeyException ,FssException{
        OutputStream os = null;
        HttpURLConnection conn = (HttpURLConnection) (new URL(_url)).openConnection();

        File f = new File(path);
        InputStream fi = new FileInputStream(f);

        conn.setRequestProperty("Content-Length", f.length() + "");
        if (_async) {
            conn.setRequestProperty("x-west-async", "true");
        }
        if (_contentMd5 != null) {
            conn.setRequestProperty("Content-MD5", _contentMd5);
        }
        if (!empty(contentType)) {
            conn.setRequestProperty("Content-Type", contentType);
        }
        if (!empty(secret)) {
            conn.setRequestProperty("Content-Secret", secret);
        }
        if (ttl > 0) {
            conn.setRequestProperty("x-west-meta-ttl", ttl + "");
        }
        if (empty(thumb)) {
            conn.setRequestProperty("x-gmkerl-thumb", thumb);
        }

        if (meta != null) {
            for (Map.Entry<String, String> entry : meta.entrySet()) {
                conn.setRequestProperty("x-west-meta-" + entry.getKey(), entry.getValue());
            }
        }

        conn.setConnectTimeout(30 * 1000);
        conn.setRequestMethod(METHOD_PUT);
        conn.setUseCaches(false);
        conn.setDoOutput(true);
        conn.setChunkedStreamingMode(0);

        if (_date != null) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            conn.setRequestProperty("Date", sdf.format(_date));
        }

        conn.setRequestProperty("Authorization", getAuthorization());
        conn.connect();
        os = conn.getOutputStream();
        byte[] data = new byte[4096];
        int temp = 0;

        // 上传文件内容
        while ((temp = fi.read(data)) != -1) {
            os.write(data, 0, temp);
        }
        // 获取返回的信息
        String result = getResponseText(conn, false, "UTF-8");

        if (os != null) {
            os.close();
        }
        if (fi != null) {
            fi.close();
        }
        if (conn != null) {
            conn.disconnect();
        }
        return result;
    }

    /**
     * 大文件分块上传
     * @param path        文件本地路径
     * @return
     * @throws Exception
     */
    public String uploadBigFile(String path) throws Exception {
        return uploadBigFile(path, 0, null, null, 0, null);
    }

    /**
     * 大文件分块上传（串行方式）
     * @param path          文件本地路径
     * @param partId        分块id，通过改参数可以实现断点续传
     * @param contentType   文件mime类型
     * @param secret        文件访问密码
     * @param ttl           文件生成时间
     * @param meta          文件元数据
     * @return
     * @throws Exception
     */
    public String uploadBigFile(String path, int partId, String contentType, String secret, int ttl, Map<String, String> meta)
        throws Exception{
        File f = new File(path);
        InputStream fi = new FileInputStream(f);
        HttpURLConnection conn = (HttpURLConnection) (new URL(_url)).openConnection();

        //初始化上传
        String result = bigFileInit(f, contentType, secret, ttl, meta);
        JSONObject jsonObject = new JSONObject(result);
        int code = jsonObject.getInt("code");
        if (code != 200) {
            throw new Exception("init upload failed!");
        }

        //执行上传
        int chunkId = partId;
        int totalChunks = (int) Math.ceil(f.length() / (double) CHUNK_SIZE );
        RandomAccessFile randomAccessFile = new RandomAccessFile(f, "r");
        System.out.println(f.length());
        try{
            while (chunkId <= totalChunks - 1) {
                if (uploadChunk(chunkId, randomAccessFile)) {
                    System.out.println("totalChunks:" + totalChunks + "  completed:" + chunkId);
                    chunkId++;
                    continue;
                }
                throw new Exception("upload part:" + chunkId + " failed");
            }
        }catch (Exception e){
            throw  e;
        }finally {
            if (randomAccessFile != null) {
                randomAccessFile.close();
            }
        }
        //完成上传
        return bigFileComplete();
    }

    /**
     * 大文件上传初始化
     * @param f
     * @param contentType
     * @param secret
     * @param ttl
     * @param meta
     * @return
     * @throws Exception
     */
    private String bigFileInit(File f, String contentType, String secret, int ttl, Map<String, String> meta) {
        Map<String, String> params = new HashMap<>();
        params.put("x-west-multi-disorder", "true");
        params.put("x-west-multi-stage", "initiate");
        params.put("x-west-multi-length", f.length() + "");

        if (!empty(contentType)) {
            params.put("x-west-multi-type", contentType);
        }
        if (!empty(secret)) {
            params.put("Content-Secret", secret);
        }
        if (ttl > 0) {
            params.put("x-west-meta-ttl", ttl + "");
        }
        if (meta != null) {
            for (Map.Entry<String, String> entry : meta.entrySet()) {
                params.put("x-west-meta-" + entry.getKey(), entry.getValue());
            }
        }
        return fssCurl(params);
    }

    /**
     * 上传第n个分块
     * @param chunkId
     * @param randomAccessFile
     * @return
     * @throws Exception
     */
    private Boolean uploadChunk(int chunkId, RandomAccessFile randomAccessFile) throws IOException,NoSuchAlgorithmException,InvalidKeyException,FssException {
        byte[] chunkBytes = readBlockByIndex(chunkId, randomAccessFile);
        Map<String, String> params = new HashMap<>();
        params.put("x-west-multi-disorder", "true");
        params.put("x-west-multi-stage", "upload");
        params.put("x-west-part-id", chunkId + "");

        OutputStream os = null;
        HttpURLConnection conn = (HttpURLConnection) (new URL(_url)).openConnection();
        conn.setConnectTimeout(30 * 1000);
        conn.setRequestMethod(METHOD_PUT);
        conn.setUseCaches(false);
        conn.setDoOutput(true);
        conn.setChunkedStreamingMode(0);

        if (_date != null) {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            conn.setRequestProperty("Date", sdf.format(_date));
        }
        for (Map.Entry<String, String> entry : params.entrySet()) {
            conn.setRequestProperty(entry.getKey(), entry.getValue());
        }

        conn.setRequestProperty("Authorization", getAuthorization());
        conn.connect();
        os = conn.getOutputStream();
        os.write(chunkBytes, 0, chunkBytes.length);
        String result = getResponseText(conn, false, "UTF-8");

        if (os != null) {
            os.close();
        }
        if (conn != null) {
            conn.disconnect();
        }
        JSONObject jsonObject = new JSONObject(result);
        int code = jsonObject.getInt("code");
        return code == 200;
    }

    /*读取第n个分片*/
    private byte[] readBlockByIndex(int index, RandomAccessFile randomAccessFile) throws IOException {
        byte[] block = new byte[CHUNK_SIZE];
        int readedSize = 0;
        int offset = index * CHUNK_SIZE;
        randomAccessFile.seek(offset);
        readedSize = randomAccessFile.read(block, 0, CHUNK_SIZE);
        //System.out.println(readedSize);

        // 如果读取到字节小于块大小说明是最后一个块
        if (readedSize < CHUNK_SIZE) {
            byte[] notFullBlock = new byte[readedSize];
            System.arraycopy(block, 0, notFullBlock, 0, readedSize);
            return notFullBlock;
        }
        return block;
    }

    /**
     * 大文件分块上传完成
     * @return
     * @throws Exception
     */
    private String bigFileComplete() throws IOException,NoSuchAlgorithmException,InvalidKeyException {
        Map<String, String> params = new HashMap<>();
        params.put("x-west-multi-disorder", "true");
        params.put("x-west-multi-stage", "complete");
        params.put("Authorization", getAuthorization());
        return fssCurl(params);
    }

    /**
     * 获取返回的header数据
     * @return
     */
    public Map<String, String> getHeaders() {
        return headers;
    }

    /*http操作*/
    private String fssCurl(Map<String, String> params) {
        return fssCurl(params, "UTF-8");
    }

    /*http操作*/
    private String fssCurl(Map<String, String> params, String charset) {
        HttpURLConnection connection = null;
        String result = null;// 返回结果字符串

        try {
            URL url = new URL(_url);
            connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod(_method);
            connection.setConnectTimeout(15000);
            connection.setReadTimeout(60000);
            connection.setUseCaches(false);
            if (!_method.equals(METHOD_DELETE) && !_method.equals(METHOD_HEAD) && !_method.equals(METHOD_GET)) {
                connection.setDoOutput(true);
                connection.setChunkedStreamingMode(0);
            }

            if (_date != null) {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                params.put("Date", sdf.format(_date));
            }
            params.put("Authorization", getAuthorization());
            for (Map.Entry<String, String> entry : params.entrySet()) {
                connection.setRequestProperty(entry.getKey(), entry.getValue());
            }
            connection.connect();
            result = getResponseText(connection, METHOD_HEAD.equals(_method), charset);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            connection.disconnect();// 关闭远程连接
        }
        return result;
    }

    /**
     * 获得连接请求的返回数据
     */
    private String getResponseText(HttpURLConnection conn, boolean isHeadMethod, String charset)
            throws IOException,FssException {

        StringBuilder text = new StringBuilder();
        InputStream is = null;
        InputStreamReader sr = null;
        BufferedReader br = null;
        int code = conn.getResponseCode();
        try {
            is = code >= 400 ? conn.getErrorStream() : conn.getInputStream();

            if (!isHeadMethod) {
                sr = new InputStreamReader(is, charset);
                br = new BufferedReader(sr);

                char[] chars = new char[4096];
                int length = 0;

                while ((length = br.read(chars)) != -1) {
                    text.append(chars, 0, length);
                }
            }
            if (code >= 200 && code < 300) {
                headers = new HashMap<>();
                Map<String, List<String>> h = conn.getHeaderFields();
                Set<String> keys = h.keySet();
                for (String key : keys) {
                    String val = conn.getHeaderField(key);
                    if (key != null && val != null) {
                        headers.put(key, val);
                    }
                }
            }
        } finally {
            if (br != null) {
                br.close();
            }
            if (sr != null) {
                sr.close();
            }
            if (is != null) {
                is.close();
            }
        }

        if (isHeadMethod) {
            if (code >= 400)
                return null;
            return "";
        }

        if (code >= 400) {
            throw new FssException(code, text.toString());
        }

        return text.toString();
    }

    private static void allowMethods(String... methods) {
        try {
            Field methodsField = HttpURLConnection.class.getDeclaredField("methods");

            Field modifiersField = Field.class.getDeclaredField("modifiers");
            modifiersField.setAccessible(true);
            modifiersField.setInt(methodsField, methodsField.getModifiers() & ~Modifier.FINAL);

            methodsField.setAccessible(true);

            String[] oldMethods = (String[]) methodsField.get(null);
            Set<String> methodsSet = new LinkedHashSet<>(Arrays.asList(oldMethods));
            methodsSet.addAll(Arrays.asList(methods));
            String[] newMethods = methodsSet.toArray(new String[0]);

            methodsField.set(null/*static field*/, newMethods);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new IllegalStateException(e);
        }
    }

    private Boolean empty(String s) {
        return s == null || s.equals("");
    }

    /*自定义异常类*/
    static class FssException extends Exception {
        int statusCode;

        public FssException(int code, String msg) {
            super("error code:" + code + "; msg:" + msg);
            statusCode = code;
        }
    }
}
